-- Data definition and setup
DECLARE @NumberOfRNs   INT
       ,@GaussianMean  FLOAT     -- For the Normal NURNs
       ,@GaussianSTDEV FLOAT
       ,@LambdaEXP     FLOAT     -- For the Exponential NURNs
       ,@WeibullAlpha  FLOAT     -- For the Weibull NURNs
       ,@WeibullBeta   FLOAT
       ,@Laplaceu      FLOAT     -- For the Laplace NURNs
       ,@Laplaceb      FLOAT

SELECT @NumberOfRNs    = 1000000
      ,@GaussianMean   = 5       -- Mean for the Normal Distribution
      ,@GaussianSTDEV  = 1.5     -- Standard Deviation for the Normal Distribution
      ,@LambdaEXP      = 1.5     -- Lambda for the Exponential Distribution
      ,@WeibullAlpha   = 1.0     -- Alpha (scale) for the Weibull Distribution
      ,@WeibullBeta    = 1.5     -- Beta (shape) for the Weibull Distribution
      ,@Laplaceu       = 4.0     -- Mu (location) for the Laplace Distribution
      ,@Laplaceb       = 1.0     -- Beta (scale) for the Laplace Distribution

--CREATE TYPE Distribution AS TABLE (EventID INT, EventProb FLOAT, CumProb FLOAT) 
DECLARE @Binomial      AS Distribution
    ,@DUniform         AS Distribution
    ,@Multinomial      AS Distribution

-- Simulate a coin toss with a Binomial Distribution
INSERT INTO @Binomial
SELECT 0, 0.5, 0.5 UNION ALL SELECT 1, 0.5, 1.0

-- Events returned by this Discrete Uniform distribution are the 6
-- Fibonacci numbers starting with the second occurrence of 1
INSERT INTO @DUniform
SELECT 1, 1./6., 1./6. UNION ALL SELECT 2, 1./6., 2./6.
UNION ALL SELECT 3, 1./6., 3./6. UNION ALL SELECT 5, 1./6., 4./6.
UNION ALL SELECT 8, 1./6., 5./6. UNION ALL SELECT 13, 1./6., 1.

-- Events returned by this Multinomial distribution are the 5
-- Mersenne primes discovered in 1952 by Raphael M. Robinson
INSERT INTO @Multinomial
SELECT 521, .10, .10 UNION ALL SELECT 607, .25, .35 UNION ALL SELECT 1279, .30, .65 
UNION ALL SELECT 2203, .15, .80 UNION ALL SELECT 2281, .2, 1.

-- Create random numbers for the selected distributions
SELECT TOP (@NumberOfRNs)
     RandomUniform     = URN
    --,RandomPoisson     = dbo.RN_POISSON(@Lambda, URN)
    ,RandomBinomial    = dbo.RN_MULTINOMIAL(@Binomial, URN)
    ,RandomDUniform    = dbo.RN_MULTINOMIAL(@DUniform, URN)
    ,RandomMultinomial = dbo.RN_MULTINOMIAL(@Multinomial, URN)
    ,RandomNormal      = dbo.RN_NORMAL(@GaussianMean, @GaussianSTDEV, URN, RAND(CHECKSUM(NEWID())))
    ,RandomExponential = dbo.RN_EXPONENTIAL(@LambdaEXP, URN)
    ,RandomWeibull     = dbo.RN_WEIBULL(@WeibullAlpha, @WeibullBeta, URN) 
    ,RandomLaplace     = dbo.RN_LAPLACE(@Laplaceu, @Laplaceb, URN)
INTO #MyRandomNumbers
FROM sys.all_columns a1 CROSS APPLY sys.all_columns a2
CROSS APPLY (SELECT RAND(CHECKSUM(NEWID()))) URN(URN)

SELECT PopulationMean=@Lambda, PopulationVAR=1./12.
    ,SampleMean=AVG(RandomUniform), SampleVAR=VAR(RandomUniform)
FROM #MyRandomNumbers

DECLARE @NoIntervals FLOAT = 40   -- Another SQL 2008 feature
-- Frequency counts for Uniform distribution
;WITH SetUpIntervals AS (
    SELECT ID=1, Prob=1.0/@NoIntervals, CumProb=1./@NoIntervals
        ,Interval=CAST('0<=U<' + CAST(1.0/@NoIntervals AS VARCHAR(8)) AS VARCHAR(40))
    UNION ALL
    SELECT ID + 1, 1.0/@NoIntervals, Cum
        ,CAST(CAST(CumProb AS VARCHAR(8)) + '<=U' + 
            CASE WHEN ID = @NoIntervals - 1 THEN '<=' ELSE '<' END +
            CAST(Cum AS VARCHAR(8)) AS VARCHAR(40))
    FROM SetUpIntervals
    CROSS APPLY (SELECT CASE WHEN ID = @NoIntervals - 1 THEN 1 ELSE CumProb+1.0/@NoIntervals END) x(Cum)
    WHERE ID < @NoIntervals)
SELECT ID, CASE WHEN ID IS NULL THEN 'Total' ELSE MAX(Interval) END AS Interval, COUNT(*) As UniformFreq
FROM #MyRandomNumbers rn
CROSS APPLY (
    SELECT TOP 1 ID, Interval
    FROM SetUpIntervals
    WHERE RandomUniform < CumProb) x
GROUP BY ID WITH ROLLUP
ORDER BY ID

-- Frequency counts for Binomial distribution
SELECT RandomBinomial, BinomialFreq, EventProb
    ,1.0*BinomialFreq/@NumberOfRNs As ActPercentOfEvents
FROM (
    SELECT RandomBinomial, COUNT(*) AS BinomialFreq
    FROM #MyRandomNumbers
    GROUP BY RandomBinomial) x
INNER JOIN @Binomial ON RandomBinomial = EventID
ORDER BY RandomBinomial

-- Frequency counts for Discrete Uniform distribution
SELECT RandomDUniform, DUniformFreq, EventProb
    ,1.0*DUniformFreq/@NumberOfRNs As ActPercentOfEvents
FROM (
    SELECT RandomDUniform, COUNT(*) AS DUniformFreq
    FROM #MyRandomNumbers
    GROUP BY RandomDUniform) x
INNER JOIN @DUniform ON RandomDUniform = EventID
ORDER BY RandomDUniform

-- Frequency counts for Multinomial distribution
SELECT RandomMultinomial, MultinomialFreq, EventProb
    ,1.0*MultinomialFreq/@NumberOfRNs As ActPercentOfEvents
FROM (
    SELECT RandomMultinomial, COUNT(*) AS MultinomialFreq
    FROM #MyRandomNumbers
    GROUP BY RandomMultinomial) x
INNER JOIN @Multinomial ON RandomMultinomial = EventID
ORDER BY RandomMultinomial

-- Compare Normal mean and standard deviation
SELECT PopulationMean=@GaussianMean, PopulationSTDEV=@GaussianSTDEV
    ,SampleMean=AVG(RandomNormal), SampleSTDEV=STDEV(RandomNormal)
FROM #MyRandomNumbers

-- Frequency counts for Normal distribution
;WITH SetUpIntervals AS (
    SELECT ID=1, StartInterval=@GaussianMean - 3 * @GaussianSTDEV
        ,EndInterval=(@GaussianMean - 3 * @GaussianSTDEV) + (6 * @GaussianSTDEV) / @NoIntervals
        ,Interval=CAST(CAST(@GaussianMean - 3 * @GaussianSTDEV AS VARCHAR) + 
            '<=U<' + CAST((@GaussianMean - 3 * @GaussianSTDEV) + (6 * @GaussianSTDEV) / @NoIntervals AS VARCHAR) AS VARCHAR(40))
    UNION ALL
    SELECT ID + 1, EndInterval, ROUND(EndInterval + (6 * @GaussianSTDEV) / @NoIntervals, 3)
        ,CAST(CAST(EndInterval AS VARCHAR) + 
            '<=U<' + CAST(ROUND(EndInterval + (6 * @GaussianSTDEV) / @NoIntervals, 3) AS VARCHAR) AS VARCHAR(40))
    FROM SetUpIntervals
    WHERE ID < @NoIntervals
)
SELECT ID=CASE WHEN z.ID IS NULL THEN y.ID ELSE z.ID END
    ,Interval=CASE WHEN z.ID IS NULL THEN 'Total' ELSE z.Interval END
    ,NormalFreq=ISNULL(NormalFreq, 0)
FROM (
    SELECT ID=CASE WHEN ID IS NULL THEN 0 ELSE ID END
        ,Interval=MAX(Interval)
        ,NormalFreq=COUNT(*)
    FROM #MyRandomNumbers rn
    CROSS APPLY (
        SELECT TOP 1 ID, Interval
        FROM SetUpIntervals
        WHERE RandomNormal < EndInterval) x
    GROUP BY ID WITH ROLLUP) y
FULL JOIN SetUpIntervals z ON z.ID = y.ID
ORDER BY ID

DECLARE @MaxEXPRN FLOAT 
-- Compare Exponential mean and variance
SELECT PopulationMean=ROUND(1./@LambdaEXP, 4)
    ,PopulationVAR=ROUND(1./SQUARE(@LambdaEXP), 4)
    ,SampleMean=ROUND(AVG(RandomExponential), 4)
    ,SampleVAR=ROUND(VAR(RandomExponential), 4)
FROM #MyRandomNumbers
SELECT @MaxEXPRN=CEILING(MAX(RandomExponential)) FROM #MyRandomNumbers

-- Frequency counts for Exponential distribution
;WITH SetUpIntervals AS (
    SELECT ID=1, StartInterval=CAST(0 AS FLOAT)
        ,EndInterval=ROUND(@MaxEXPRN / @NoIntervals, 3)
        ,Interval=CAST( 
            '0<=U<' + CAST(ROUND(@MaxEXPRN / @NoIntervals, 3) AS VARCHAR) AS VARCHAR(40))
    UNION ALL
    SELECT ID + 1, EndInterval, ROUND(EndInterval + @MaxEXPRN / @NoIntervals, 3)
        ,CAST(CAST(EndInterval AS VARCHAR) + 
            '<=U<' + CAST(ROUND(EndInterval + @MaxEXPRN / @NoIntervals, 3) AS VARCHAR) AS VARCHAR(40))
    FROM SetUpIntervals
    WHERE ID < @NoIntervals
)
SELECT ID=CASE WHEN z.ID IS NULL THEN y.ID ELSE z.ID END
    ,Interval=CASE WHEN z.ID IS NULL THEN 'Total' ELSE z.Interval END
    ,ExponentialFreq=ISNULL(ExponentialFreq, 0)
FROM (
    SELECT ID=CASE WHEN ID IS NULL THEN 0 ELSE ID END
        ,Interval=MAX(Interval)
        ,ExponentialFreq=COUNT(*)
    FROM #MyRandomNumbers rn
    CROSS APPLY (
        SELECT TOP 1 ID, Interval
        FROM SetUpIntervals
        WHERE RandomExponential < EndInterval) x
    GROUP BY ID WITH ROLLUP) y
FULL JOIN SetUpIntervals z ON z.ID = y.ID
ORDER BY ID

DECLARE @MAXWB FLOAT, @MINWB FLOAT
-- Setup MAX/MIN for calculating intervals
SELECT @MAXWB=CEILING(MAX(RandomWeibull)), @MINWB=FLOOR(MIN(RandomWeibull))
FROM #MyRandomNumbers
 
-- Frequency counts for Weibull distribution
;WITH SetUpIntervals AS (
    SELECT ID=1, StartInterval=@MINWB
        ,EndInterval=ROUND(@MINWB + (@MAXWB - @MINWB) / @NoIntervals, 3)
        ,Interval=CAST(CAST(@MINWB AS VARCHAR) +  
            '<=U<' + CAST(ROUND((@MAXWB - @MINWB) / @NoIntervals, 3) AS VARCHAR) AS VARCHAR(40))
    UNION ALL
    SELECT ID + 1, EndInterval, ROUND(EndInterval + (@MAXWB - @MINWB) / @NoIntervals, 3)
        ,CAST(CAST(EndInterval AS VARCHAR) + 
            '<=U<' + CAST(ROUND(EndInterval + (@MAXWB - @MINWB) / @NoIntervals, 3) AS VARCHAR) AS VARCHAR(40))
    FROM SetUpIntervals
    WHERE ID < @NoIntervals
)
SELECT ID=CASE WHEN z.ID IS NULL THEN y.ID ELSE z.ID END
    ,Interval=CASE WHEN z.ID IS NULL THEN 'Total' ELSE z.Interval END
    ,WeibullFreq=ISNULL(WeibullFreq, 0)
FROM (
    SELECT ID=CASE WHEN ID IS NULL THEN 0 ELSE ID END
        ,Interval=MAX(Interval)
        ,WeibullFreq=COUNT(*)
    FROM #MyRandomNumbers rn
    CROSS APPLY (
        SELECT TOP 1 ID, Interval
        FROM SetUpIntervals
        WHERE RandomWeibull < EndInterval) x
    GROUP BY ID WITH ROLLUP) y
FULL JOIN SetUpIntervals z ON z.ID = y.ID
ORDER BY ID

DECLARE @MAXLaplace FLOAT, @MINLaplace FLOAT
-- Compare Laplace mean and variance
SELECT PopulationMean=@Laplaceu
    ,PopulationVAR=2*SQUARE(@Laplaceb)
    ,SampleMean=ROUND(AVG(RandomLaplace), 4)
    ,SampleVAR=ROUND(VAR(RandomLaplace), 4)
FROM #MyRandomNumbers

-- Setup MAX/MIN for calculating intervals
SELECT @MAXLaplace=CEILING(MAX(RandomLaplace)), @MINLaplace=FLOOR(MIN(RandomLaplace))
FROM #MyRandomNumbers

-- Frequency counts for Laplace distribution
;WITH SetUpIntervals AS (
    SELECT ID=1, StartInterval=@MINLaplace
        ,EndInterval=ROUND(@MINLaplace + (@MAXLaplace - @MINLaplace) / @NoIntervals, 3)
        ,Interval=CAST(CAST(@MINLaplace AS VARCHAR) +  
            '<=U<' + CAST(ROUND(@MINLaplace + (@MAXLaplace - @MINLaplace) / @NoIntervals, 3) AS VARCHAR) AS VARCHAR(40))
    UNION ALL
    SELECT ID + 1, EndInterval, ROUND(EndInterval + (@MAXLaplace - @MINLaplace) / @NoIntervals, 3)
        ,CAST(CAST(EndInterval AS VARCHAR) + 
            '<=U<' + CAST(ROUND(EndInterval + (@MAXLaplace - @MINLaplace) / @NoIntervals, 3) AS VARCHAR) AS VARCHAR(40))
    FROM SetUpIntervals
    WHERE ID < @NoIntervals
)
SELECT ID=CASE WHEN z.ID IS NULL THEN y.ID ELSE z.ID END
    ,Interval=CASE WHEN z.ID IS NULL THEN 'Total' ELSE z.Interval END
    ,LaplaceFreq=ISNULL(LaplaceFreq, 0)
FROM (
    SELECT ID=CASE WHEN ID IS NULL THEN 0 ELSE ID END
        ,Interval=MAX(Interval)
        ,LaplaceFreq=COUNT(*)
    FROM #MyRandomNumbers rn
    CROSS APPLY (
        SELECT TOP 1 ID, Interval
        FROM SetUpIntervals
        WHERE RandomLaplace < EndInterval) x
    GROUP BY ID WITH ROLLUP) y
FULL JOIN SetUpIntervals z ON z.ID = y.ID
ORDER BY ID

DROP TABLE #MyRandomNumbers
--DROP FUNCTION RN_EXPONENTIAL, RN_LAPLACE, RN_MULTINOMIAL, RN_NORMAL, RN_WEIBULL
--DROP TYPE Distribution
